#######################################################################
# Compute consistent classical or robust multivariate coefficient of variation estimates
# under: 
# - normality
# - multivariate Student 
# - multivariate power exponential
# /!\ The elliptical distributions are defined such that COV(X)= sigma, 
#######################################################################
library(robustbase)
library(normalp)

# Files needed to run this script
# - MCDconsist
# - Sconsist
# - fastS_normality
# - fastS_consistency

# ------------ Main functions

MCV<- function(data, estim='class', bdp=0.25, dist= "norm",df=NULL){
  # Function to compute consistent MCV estimates 
  
  # INPUTS:
  # data: data matrix of dimension n times p
  # estim : location and scatter estimators 
  #     'class' : sample estimate
  #     'MCD'   : MCD estimate (raw)
  #     'RMCD'  : one-step reweighted MCD estimate
  #     'S'     : S estimate with Tukey's biweight function
  # bdp : breakdown point for the MCD, RMCD or S estimators
  # dist : assumed distribution used for the computation of consistency factors 
  #     'norm' : multivariate normal
  #     'stud' : multivariate Student
  #     'powerexp' : multivariate power exponential distribution
  #     /!\ The elliptical distributions are defined such that COV(X)= sigma, 
  # df : degree of freedom for the Student or powerexp distribution 
  #       NB: for the power exp, df=1 corresponds to the normal distribution
  # OUTPUTS:
  # R = MCV Reyment
  # VV = MCV Van Valen
  # VN = MCV Voinov and Nikulin
  # AZ = MCV Albert and Zhang
  
  p=ncol(data)
  n=nrow(data)
  
  if(n<p) stop("the sample size needs to be larger than the dimension")
  if(p<2) stop("you are in the univariate setting, compute the univariate CV")
  if((dist=="stud" | dist=="powerexp") & is.null(df)) stop("you have to specify a degree of freedom")
  
  # Consistent location and scatter parameters
    if(estim=="class"){
      mean<- apply(data,2,mean)
      cov<- ((n-1)/n ) *cov(data)
      
    }else if(estim=="MCD" || estim=="RMCD"){
      
      # raw MCD
      MCDraw<-consistMCD(data=data, dist=dist, df=df,bdp=bdp)
      mean<- MCDraw$center
      cov<-MCDraw$cov
      
      # reweighted MCD
     if(estim=="RMCD"){
       RewMCD<-consistRMCD(data=data,moy0=mean, cov0=cov, dist=dist, df=df)
       mean<- RewMCD$center
       cov<-RewMCD$cov
     }
    }else if(estim=="S"){
      Sestim<- consistS(data=data,dist=dist,df=df,bdp=bdp)
      mean<- Sestim$center
      cov<- Sestim$cov
    }else stop("This is not  valid estim value. The possible values are: 'class', 'MCD', 'RMCD' or 'S'")
 return(list(R= cvreyment(mean,cov), VV=cvvanvalen(mean,cov), VN= cvvoinov(mean,cov), AZ=cvaz(mean,cov)))
}


# Univariate CV
uniCV<-function(data, estim='class', dist= "norm",df=NULL ){
  # INPUT:
  # data : data vector
  # estim : location and scatter estimators 
  #     'class' : sample estimate
  #     'MCD'   : MCD estimate (raw)
  #     'RMCD'  : one-step reweighted MCD estimate
  #     'S'     : S estimate with Tukey's biweight function
  # bdp : breakdown point for the MCD, RMCD or S estimators
  # dist : assumed distribution used for the computation of consistency factors 
  #     'norm' : multivariate normal
  #     'stud' : multivariate Student
  #     'powerexp' : multivariate power exponential distribution
  #     /!\ The elliptical distributions are defined such that COV(X)= sigma, 
  # df : degree of freedom for the Student or powerexp distribution 
  #       NB: for the power exp, df=1 corresponds to the normal distribution
  
  # OUTPUT :
  # univariate coefficient of variation
  
  data<-as.matrix(data)
  p<-ncol(data)
  n<-nrow(data)
  
  if(estim=="class"){
    mu<-mean(data)
    sd<-sqrt(n-1)/sqrt(n)*sd(data)
    
  }else if(estim=="IQRmed"){ 
    if(dist=="norm"){
      consist<-1/(2*qnorm(0.75))
    }else if(dist=="stud"){
      consist<-1/(2*sqrt((df-2)/df)*qt(0.75,df))
    }else if(dist=="powerexp"){
      a<-(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
      consist<- 1/(2*qnormp(0.75,mu=0,sigmap=1/(2*a*df)^(1/(2*df)), p=2*df))
    }
    mu<-median(data)
    sd<- consist*(quantile(data,probs=c(0.75))-quantile(data,probs=c(0.25)))
    
  }else if(estim=="MADmed"){
    if(dist=="norm"){
      consist<- 1/qnorm(0.75)
    }else if(dist=="stud"){
      consist<- 1/(sqrt((df-2)/df)* qt(0.75,df))
    }else if(dist=="powerexp"){
      a<-(gamma((p+2)/(2*df))/(p*gamma(p/(2*df))))^df
      consist<- 1/qnormp(0.75,mu=0,sigmap=1/(2*a*df)^(1/(2*df)), p=2*df)
    }
    mu<-median(data)
    sd<- consist*median(abs(data-median(data)))
  }
  return(sd/mu)
}



#------------------- Auxiliary functions

cvreyment<-function(mu,sigma){
  p<-length(mu)
  return( sqrt( det(sigma)^(1/p) / (mu%*%mu) ) )
}
cvvanvalen<-function(mu,sigma){
  return(sqrt(sum(diag(sigma))/(mu%*%mu)))
}
cvvoinov<-function(mu,sigma){
  return(1/sqrt(mu %*% solve(sigma)%*%mu) )
}
cvaz<-function(mu,sigma){
  return( sqrt( (mu%*% sigma %*% mu) / (mu%*%mu)^2 ))
}



